
/* 
 * Space Rivals v1.0 - simple Amstrad CPC 4-player game
 * Developed by LuBlu Entertainment for the CPC Retro Dev Compo 2015
 * Using CPCtelera v1.1.1
 * 
 * Be sure to check out our other projects too:
 * 
 * PirateDiamonds.com - our first Android game! Challenge your friends! 
 * IndieGameMusic.com - find the music you need for your own game projects!
 * 
 * Also keep an eye out for "8bit Stories" - a diskmag-like project containing
 * lots of stories about the computers I remember from my childhood.
 * (I.e. mostly about the Amstrad CPC).
 * To be released sometime in 2016/2017 (hopefully not later).
 * More info at IndieGameMusic.com/8bit-stories
 *
 *
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <cpctelera.h>
#include <string.h>
#include "sprites.h"
#include "titlemusic.h"

#define SCROLLCHARPOS 0xc7ce
#define CONTROLS_UNAVAILABLE 0
#define CONTROLS_QAOP 1
#define CONTROLS_JOY0 2
#define CONTROLS_JOY1 4
#define CONTROLS_MULTIPLAYA 8
#define CONTROLS_MULTIPLAYB 16
#define TILEWIDTH 4
#define TILEHEIGHT 16
#define STARTY 168
#define TOPY 25
#define DEAD 0
#define ALIVE 1
#define DYING 2
#define PLAYER1 0
#define PLAYER2 1
#define PLAYER3 2
#define PLAYER4 3
#define STARTENERGY 6
#define STARTBULLETS 30
#define STARTSHIELD 6
#define KEY_UP 1
#define KEY_DOWN 2
#define KEY_LEFT 4
#define KEY_RIGHT 8
#define KEY_FIRE 16
#define KEY_FIRE2 32
#define POWERUP_SHIELD 1
#define POWERUP_ENERGY 2
#define POWERUP_BULLETS 3
#define POWERUP_NONE 4
#define MAX_SIMULTANEOUS_BULLETS 8
#define SHOT_PAUSE 14

extern u8 readPort(u16 port);
const u16 BULLETSADDRESS[4] = {0xd854, 0xd868, 0xd87c, 0xd890};
const u16 SHIELDADDRESS[4] = {0xd804, 0xd818, 0xd82c, 0xd840};
const u16 ENERGYADDRESS[4] = {0xd80d, 0xd821, 0xd835, 0xd849};
const u8 speedshifter = 3;
const i8 MAXUPSPEED = 32, MAXDOWNSPEED = 48;
const u8 xposition[4] = {8, 28, 48, 68};
const u8 audiochannel[4] = {AY_CHANNEL_A, AY_CHANNEL_B, AY_CHANNEL_B, AY_CHANNEL_C};
u8 count;
u8 yposition[4]; // Player y position
i8 vector[4]; // Player vector (speed)
i8 clear[4]; // Number of lines to clear above/below player-sprite
u8 energy[4]; // Player energy
u8 bullets[4]; // Player bullets
u8 shield[4]; // Player shield
u8 shieldcounter[4];
u8 box[4];
u8 carryingBox[4];
u8 status[4]; // Player status (ALIVE, DYING or DEAD)
u8 explosionframe[4];
u8 ouch[4]; // Player ouch counters
u8 shotmoved[4][MAX_SIMULTANEOUS_BULLETS]; // Number of bytes a shot has moved since fired
u8 shotdirection[4][MAX_SIMULTANEOUS_BULLETS]; // Direction of shot. 1 = left, 2 = right
u8 shotx[4][MAX_SIMULTANEOUS_BULLETS];
u8 shoty[4][MAX_SIMULTANEOUS_BULLETS];
u8* shotaddress[4][MAX_SIMULTANEOUS_BULLETS];
u8 shotiterator[4];
u8 shotpause[4];
u8 storage[4];
u8 menuscreen;
u8 weHaveAWinner;
u8* pointer;
u8 n, t, a;
u8 playercontrols[4];
u8 controlsum;

void AxelayScroller() {
  __asm
    ld hl,#0xc7cf; start from right
    ld c,#8 ; pixel lines
_Scroll_loop_outer::
    ld b,#80 ; bytes in a line
    ld d,#0 ; clear incoming buffer on right side of screen
_Scroll_loop_inner::
    ld a,(hl)
    ld e,a ; save current data for later
    and a,#0x77 ; got 3 right pixels
    rlca ; move them left
    or a,d ; get new right pixel from previous bytes left pixel, stored below
    ld (hl),a ; write back scrolled byte
    ld a,e ; get old byte data back to a
    and a,#0x88 ; get the left pixel
    rrca
    rrca
    rrca ; put in position of right pixel for byte to left of this one
    ld d,a ; and save in d for next byte
    dec hl ; move to left byte of current one
    djnz _Scroll_loop_inner
; next pixel line
    ld de,#0x800+#80
    add hl,de ; got hl pointing to right side of screen, 1 line down
    dec c ; count down pixel line count
    jr nz,_Scroll_loop_outer
__endasm;
}

void drawEnergy(u8 player) {
  cpct_drawSolidBox((u8*) ENERGYADDRESS[player], 0, STARTENERGY, 5);
  if (energy[player]) cpct_drawSolidBox((u8*) ENERGYADDRESS[player], 6, energy[player], 5);
}

void drawShield(u8 player) {
  cpct_drawSolidBox((u8*) SHIELDADDRESS[player], 0, STARTSHIELD, 5);
  if (shield[player]) cpct_drawSolidBox((u8*) SHIELDADDRESS[player], 6, shield[player], 5);
}

void drawBullets(u8 player) {
  cpct_drawSolidBox((u8*) BULLETSADDRESS[player], 0, (STARTBULLETS >> 1), 5);
  if ((bullets[player] >> 1) > 0) cpct_drawSolidBox((u8*) BULLETSADDRESS[player], 6, (bullets[player] >> 1), 5);
}

void ouchie(u8 player) {
  if (energy[player] == 0) {
    explosionframe[player] = 0;
    status[player] = DYING;
    cpct_akp_SFXPlay(2, 15, 40, 0, 0, audiochannel[player]);
  } else {
    energy[player]--;
    ouch[player] = 10;
    drawEnergy(player);
    cpct_akp_SFXPlay(3, 15, 40, 0, 0, audiochannel[player]);
  }
}

void initialize(u8 player) {
  yposition[player] = STARTY;
  vector[player] = 0;
  clear[player] = 0;
  energy[player] = STARTENERGY;
  bullets[player] = STARTBULLETS;
  shield[player] = STARTSHIELD;
  shieldcounter[player] = 0;
  shotiterator[player] = 0;
  shotpause[player] = 0;
  status[player] = ALIVE;
  explosionframe[player] = 0;
  box[player] = 0;
  carryingBox[player] = 0;
  storage[player] = 198;
  if (!menuscreen) {
    cpct_drawSprite(rocket[player], row[player][yposition[player]], TILEWIDTH, TILEHEIGHT);
    drawEnergy(player);
    drawShield(player);
    drawBullets(player);
  }
  if (player == PLAYER1) {
    cpct_drawStringM1_f("     ", (u8*) 0xc3c5, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc415, 2, 0);
  } else if (player == PLAYER2) {
    cpct_drawStringM1_f("     ", (u8*) 0xc3d9, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc429, 2, 0);
  } else if (player == PLAYER3) {
    cpct_drawStringM1_f("     ", (u8*) 0xc3ed, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc43d, 2, 0);
  } else if (player == PLAYER4) {
    cpct_drawStringM1_f("     ", (u8*) 0xc401, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc451, 2, 0);
  }
}

u8 getControlState(u8 player) {
  u8 result = 0;
  if (playercontrols[player] == CONTROLS_QAOP) {
    if (cpct_isKeyPressed(Key_Q)) result |= KEY_UP;
    //if (cpct_isKeyPressed(Key_A)) result |= KEY_DOWN;
    if (cpct_isKeyPressed(Key_O)) result |= KEY_LEFT;
    if (cpct_isKeyPressed(Key_P)) result |= KEY_RIGHT;
    if (cpct_isKeyPressed(Key_A)) result |= KEY_FIRE;
    if (cpct_isKeyPressed(Key_Space)) result |= KEY_FIRE;
  } else if (playercontrols[player] == CONTROLS_MULTIPLAYA) {
    result = readPort(0xf990); // Get Multiplay A
  } else if (playercontrols[player] == CONTROLS_MULTIPLAYB) {
    result = readPort(0xf991); // Get Multiplay B
  } else if (playercontrols[player] == CONTROLS_JOY0) {
    if (cpct_isKeyPressed(Joy0_Up)) result |= KEY_UP;
    if (cpct_isKeyPressed(Joy0_Down)) result |= KEY_DOWN;
    if (cpct_isKeyPressed(Joy0_Left)) result |= KEY_LEFT;
    if (cpct_isKeyPressed(Joy0_Right)) result |= KEY_RIGHT;
    if (cpct_isKeyPressed(Joy0_Fire1)) result |= KEY_FIRE;
    if (cpct_isKeyPressed(Joy0_Fire2)) result |= KEY_FIRE2;
  } else if (playercontrols[player] == CONTROLS_JOY1) {
    if (cpct_isKeyPressed(Joy1_Up)) result |= KEY_UP;
    if (cpct_isKeyPressed(Joy1_Down)) result |= KEY_DOWN;
    if (cpct_isKeyPressed(Joy1_Left)) result |= KEY_LEFT;
    if (cpct_isKeyPressed(Joy1_Right)) result |= KEY_RIGHT;
    // NOTE: The following two lines switch the FIRE buttons arround due to a bug in CPCtelera 1.1.1 (I think)
    // Should most probably be switched around again if compiling with a newer version of CPCtelera.
    if (cpct_isKeyPressed(Joy1_Fire2)) result |= KEY_FIRE;
    if (cpct_isKeyPressed(Joy1_Fire1)) result |= KEY_FIRE2;
  }
  return result;
}

void newBox(u8 player) {
  u8 c = (count + player * 5) % 50;
  if (c < 10) { // Powerup
    if (bullets[player] < 6) box[player] = POWERUP_BULLETS;
    else if (shield[player] < 2) box[player] = POWERUP_SHIELD;
    else if (energy[player] < 2) box[player] = POWERUP_ENERGY;
    else box[player] = POWERUP_NONE;
  } else box[player] = POWERUP_NONE;
  cpct_drawSprite(powerup[box[player] - 1], row[player][27], TILEWIDTH, TILEHEIGHT);
}

void getBox(u8 player) {
  cpct_drawSolidBox(row[player][27], 0, TILEWIDTH, TILEHEIGHT);
  carryingBox[player] = 1;
  if (box[player] == POWERUP_BULLETS) {
    bullets[player] = STARTBULLETS;
    drawBullets(player);
  } else if (box[player] == POWERUP_ENERGY) {
    energy[player] = STARTENERGY;
    drawEnergy(player);
  } else if (box[player] == POWERUP_SHIELD) {
    shield[player] = STARTSHIELD;
    drawShield(player);
  }
  cpct_akp_SFXPlay(6, 15, 40, 0, 0, audiochannel[player]);
}

void handleLogic(u8 player) {
  u8 controlstate = getControlState(player);
  if (status[player] == ALIVE) {
    if ((controlstate & KEY_FIRE2) || (controlstate & KEY_UP)) {
      vector[player] -= 2;
    }
    if ((controlstate & KEY_FIRE) && shield[player] > 0 && shieldcounter[player] == 0) {
      shieldcounter[player] = 50;
      shield[player]--;
      drawShield(player);
    }
    if ((controlstate & KEY_LEFT) && shotpause[player] > SHOT_PAUSE && bullets[player] > 0) {
      shotpause[player] = 0;
      shotiterator[player]++;
      if (shotiterator[player] == MAX_SIMULTANEOUS_BULLETS) shotiterator[player] = 0;
      shotx[player][shotiterator[player]] = xposition[player] - 1;
      shoty[player][shotiterator[player]] = yposition[player] + 7;
      shotaddress[player][shotiterator[player]] = row[player][yposition[player] + 7];
      shotdirection[player][shotiterator[player]] = 1;
      shotmoved[player][shotiterator[player]] = 0;
      bullets[player]--;
      cpct_akp_SFXPlay(1, 15, 40, 0, 0, audiochannel[player]);
      drawBullets(player);
    }
    if ((controlstate & KEY_RIGHT) && shotpause[player] > SHOT_PAUSE && bullets[player] > 0) {
      shotpause[player] = 0;
      shotiterator[player]++;
      if (shotiterator[player] == MAX_SIMULTANEOUS_BULLETS) shotiterator[player] = 0;
      shotx[player][shotiterator[player]] = xposition[player] + 4;
      shoty[player][shotiterator[player]] = yposition[player] + 7;
      shotaddress[player][shotiterator[player]] = row[player][yposition[player] + 7] + 3;
      shotdirection[player][shotiterator[player]] = 2;
      shotmoved[player][shotiterator[player]] = 0;
      bullets[player]--;
      cpct_akp_SFXPlay(1, 15, 40, 0, 0, audiochannel[player]);
      drawBullets(player);
    }
    vector[player]++;
    if (vector[player] > MAXDOWNSPEED) vector[player] = MAXDOWNSPEED;
    if (vector[player]<-MAXUPSPEED) vector[player] = -MAXUPSPEED;
    clear[player] = (vector[player] >> speedshifter);
    yposition[player] += clear[player];
    if (yposition[player] < 43 && carryingBox[player] == 0 && box[player] > 0) getBox(player);
    else if (yposition[player] < TOPY) { // Player has hit the top
      yposition[player] = TOPY;
      vector[player] = 0;
      if (clear[player]<-1 && ouch[player] == 0) if (shieldcounter[player] == 0) ouchie(player);
        else cpct_akp_SFXPlay(4, 15, 40, 0, 0, audiochannel[player]);
    } else if (yposition[player] > STARTY) { // Player has hit the ground
      yposition[player] = STARTY;
      vector[player] = 0;
      if (clear[player] > 1 && ouch[player] == 0) if (shieldcounter[player] == 0) ouchie(player);
        else cpct_akp_SFXPlay(4, 15, 40, 0, 0, audiochannel[player]);
      if (carryingBox[player] > 0) {
        cpct_drawSprite(line, row[player][storage[player]], TILEWIDTH, 1);
        storage[player]--;
        carryingBox[player] = 0;
        box[player] = 0;
        if (storage[player] == 184 && status[player] == ALIVE) { // Win if player didn't land too hard and died
          weHaveAWinner = player + 1;
        }
      }
    }
    if (clear[player] < 0) cpct_drawSolidBox(row[player][yposition[player] + 16], 0, 4, -clear[player]);
    else if (clear[player] > 0) cpct_drawSolidBox(row[player][yposition[player] - clear[player]], 0, 4, clear[player]);
    if (ouch[player]) {
      if (count % 6 < 3) cpct_drawSprite(rocket[player], row[player][yposition[player]], TILEWIDTH, TILEHEIGHT);
      else cpct_drawSolidBox(row[player][yposition[player]], 0, TILEWIDTH, TILEHEIGHT);
      ouch[player]--;
    } else {
      if (shieldcounter[player]) {
        if (count % 4 < 2) cpct_drawSprite(rocketshield1[player], row[player][yposition[player]], TILEWIDTH, TILEHEIGHT);
        else cpct_drawSprite(rocketshield2[player], row[player][yposition[player]], TILEWIDTH, TILEHEIGHT);
        shieldcounter[player]--;
      } else {
        cpct_drawSprite(rocket[player], row[player][yposition[player]], TILEWIDTH, TILEHEIGHT);
      }
    }
    if (box[player] == 0 && carryingBox[player] == 0) {
      newBox(player);
    }
  } else if (status[player] == DYING) { // Player 1 is exploding
    cpct_drawSprite(explosion[explosionframe[player]], row[player][yposition[player]], TILEWIDTH, TILEHEIGHT);
    if (explosionframe[player] < 4) cpct_setPalette(explosionpalette[explosionframe[player]], 4);
    explosionframe[player]++;
    if (explosionframe[player] == 22) {
      status[player] = DEAD;
      if (status[PLAYER1] + status[PLAYER2] + status[PLAYER3] + status[PLAYER4] == 1) { // We have a last man standing = winner
        // Who?
        for (n = 0; n < 4; n++) if (status[n] == 1) weHaveAWinner = n + 1;
      }
    }
  }
}

void handleShot(u8 player) {
  u8 hit;
  hit = 0;
  if (shotpause[player] < 250) shotpause[player]++;
  for (t = 0; t < MAX_SIMULTANEOUS_BULLETS; t++) {
    if (shotdirection[player][t]) {
      pointer = (u8*) shotaddress[player][t];
      *pointer = 0; // Clear last position
      if (shotdirection[player][t] == 1) { // Move shot left
        shotaddress[player][t]--;
        shotx[player][t]--;
      } else if (shotdirection[player][t] == 2) { // Move shot right
        shotaddress[player][t]++;
        shotx[player][t]++;
      }
      if (shotx[player][t] == 80) { // Shot has reached right side of the screen
        shotx[player][t] = 0;
        shotaddress[player][t] -= 79;
      } else if (shotx[player][t] == 255) { // shot has reached left side of the screen
        shotx[player][t] = 79;
        shotaddress[player][t] += 79;
      }
      pointer = (u8*) shotaddress[player][t];
      if (*pointer > 0) { // Something was hit!
        for (a = 0; a < 4; a++) { // Loop through all players
          if (shotx[player][t] >= xposition[a] && shotx[player][t] <= xposition[a] + 4 && shoty[player][t] >= yposition[a] && shoty[player][t] <= yposition[a] + 15) { // Player a was hit
            hit = a + 1;
            if (shieldcounter[a] > 0) { // But he had shield on. Bounce the shot
              if (shotdirection[player][t] == 1) {
                shotdirection[player][t] = 2;
                shotx[player][t] += 2;
                shotaddress[player][t] += 2;
              } else {
                shotdirection[player][t] = 1;
                shotx[player][t] -= 2;
                shotaddress[player][t] -= 2;
              }
              cpct_akp_SFXPlay(4, 15, 40, 0, 0, audiochannel[a]);
            } else { // Player a was hit and he didn't have shield on
              shotdirection[player][t] = 0; // Kill the shot
              ouchie(a); // Lose energy
            }
            break; // No need to search other players
          }
        }
        if (hit == 0) shotdirection[player][t] = 0; // Bullet hit the scenary. Kill it.
      } else { // Nothing was hit, move shot to next byte
        shotmoved[player][t]++;
        if (shotmoved[player][t] == 68) shotdirection[player][t] = 0;
        else {
          if (shotmoved[player][t] % 8 < 4) {
            *pointer = 255;
          } else {
            *pointer = 15;
          }
        }
      }
    }
  }
}

void screen1(u8 * sprite) {
  cpct_drawSprite(sprite, (u8*) 0xe0ff, 10, 24);
}

void screen2(u8 * sprite) {
  cpct_drawSprite(sprite, (u8*) 0xe113, 10, 24);
}

void screen3(u8 * sprite) {
  cpct_drawSprite(sprite, (u8*) 0xe127, 10, 24);
}

void checkJoins() {
  if (a < 4) {
    if (!(controlsum & CONTROLS_JOY0) && (cpct_isKeyPressed(Joy0_Fire1) || cpct_isKeyPressed(Joy0_Fire2))) {
      playercontrols[a] = CONTROLS_JOY0;
      controlsum |= CONTROLS_JOY0;
      initialize(a);
      a++;
      menuscreen = 0;
    }
    if (!(controlsum & CONTROLS_JOY1) && (cpct_isKeyPressed(Joy1_Fire1) || cpct_isKeyPressed(Joy1_Fire2))) {
      playercontrols[a] = CONTROLS_JOY1;
      controlsum |= CONTROLS_JOY1;
      initialize(a);
      a++;
      menuscreen = 0;
    }
    if (!(controlsum & CONTROLS_MULTIPLAYA) && ((readPort(0xf990) & 0x3f) == 16 || (readPort(0xf990) & 0x3f) == 32)) {
      playercontrols[a] = CONTROLS_MULTIPLAYA;
      controlsum |= CONTROLS_MULTIPLAYA;
      initialize(a);
      a++;
      menuscreen = 0;
    }
    if (!(controlsum & CONTROLS_MULTIPLAYB) && ((readPort(0xf991) & 0x3f) == 16 || (readPort(0xf991) & 0x3f) == 32)) {
      playercontrols[a] = CONTROLS_MULTIPLAYB;
      controlsum |= CONTROLS_MULTIPLAYB;
      initialize(a);
      a++;
      menuscreen = 0;
    }
    cpct_scanKeyboard();
    if (!(controlsum & CONTROLS_QAOP) && (cpct_isKeyPressed(Key_Space) || cpct_isKeyPressed(Key_Return))) {
      playercontrols[a] = CONTROLS_QAOP;
      controlsum |= CONTROLS_QAOP;
      initialize(a);
      a++;
      menuscreen = 0;
    }
  }
}

void main(void) {
  u8 pause = 0;
  u8 normalspeed = 1; // 1 for normal speed. 0 for turbo. Can be changed in title-screen with T and N keys.
  const u8 *text = "        Welcome to Space Rivals!        $         Developed by LuBlu Entertainment in July-October 2015 for the CPC Retro Dev competition.             Instructions:   Be the first to bring 14 crates to planet Vpac7 in your rocket.    Press FIRE2 or UP to thrust your rocket.   Press RIGHT/LEFT to shoot right and left.   Press FIRE1 to activate shield.            Title-screen options:   Press T to activate TURBO MODE.   Press N to go back to NORMAL MODE.                    Credits:   Code & music/sfx by Mr.Lou.     Graphics & illustration by BlueAngel.                    Thanks to ronaldo and arnoldemu for lots of help and advice with the code.     Thanks to Axelay for the scrolltext routine.     Thanks to TotO for lending us the MultiPlay prototype.                Space Rivals was developed using CPCtelera v1.1.1.    Music was composed with STarKos v1.21.    Graphics was created with Adobe Photoshop and converted to CPC format with Retro Game Asset Studio.                                        Space Rivals was inspired by the Philips Videopac G7000 game \"Space Rendezvous\", that I used to play as a kid.     The 1st rocket sprite is a tribute to that game.     The 2nd rocket sprite is a tribute to the Amstrad CPC; representing symbol 239 in the CPC ASCII table, that I used a lot later in my childhood when coding my amateurish BASIC stuff.                             Ok, I'd like to advertise a bit now for my project \"8bit Stories\" that I have worked on for several years.  It is a collection of stories about the time I spent with the CPC (and other computers from that time) in my childhood, presented like a diskmag and running on Blu-ray players.    Keep yourself updated on IndieGameMusic.com/8bit-stories                         Alright...     time for some greetings.           Greetings to Mr.Fox!          $Hey cousin!   Good to see you here on the CPC again!   Can you believe it's been 25 years since we had our CPC era??   You know, as I'm writing this I've just finished looking through all of your old CPC tapes that you gave me back in 2002 or so.  I found an old demo on one of the tapes from 1988 that you made.  Great nostalgic stuff!  :-)     Be sure to check out \"8bit Stories\" for a lot more nostalgia!  :-)                  Greetings to N.W.C.!          $We are waiting for your return to the CPC, and (still) curious to see that game you were working on back in the day.    Thanks a lot for all your help through-up the years regarding recovery of my lost files!  Much appreciated!  :-)                   And greetings to everyone at CPCwiki.eu and pushnpop.net who has helped and supported me with various CPC related ideas and projects through-up the years!           And finally lots of greetings to everyone doing CPC projects!              End of greetings....                      Be sure to check out our mobile game too!   It runs on Android and older JavaME enabled phones (including non-touchscreen ones) and can be downloaded from www.PirateDiamonds.com       Go challenge your friends!   :-)                                        $ $Still here, eh?     Why not start your own game-project?   If you have a little coding experience then CPCtelera is a great framework for doing CPC games!                    If you're creating a game on another platform then my site www.IndieGameMusic.com is a good resource for music!   Lots of different styles and filetypes there from lots of different musicians.  It offers lots of search options - and the prices are great too!                                         $ $   ";
  u16 textlen = strlen(text);
  u16 nextChar = 0;

  cpct_disableFirmware();
  cpct_fw2hw(palette, 4);
  cpct_fw2hw(explosionpalette, 16);
  cpct_setVideoMode(1);
  cpct_setPalette(palette, 4);
  cpct_setBorder(palette[0]);

  while (1) {

    nextChar = 0;
    controlsum = 0;
    a = 0;
    // Start game
    cpct_clearScreen(0);
    cpct_drawSprite(title, (u8*) 0xc00a, 63, 96);
    cpct_drawSprite(star1, (u8*) 0xc000, 1, 3);
    cpct_drawSprite(star2, (u8*) 0xc84e, 2, 5);
    cpct_drawSprite(star3, (u8*) 0xc192, 2, 5);
    *((u8*) 0xc0f3) = 2; // Pen 2
    *((u8*) 0xc04c) = 2; // Pen 2
    cpct_drawSprite(star1, (u8*) 0xc271, 1, 3);
    *((u8*) 0xc280) = 32; // Pen 1
    *((u8*) 0xc31a) = 32; // Pen 1
    *((u8*) 0xc415) = 2; // Pen 2
    cpct_drawSprite(star3, (u8*) 0xccc7, 2, 5);
    menuscreen = 1;

    // Reset shots and ouch for all players
    for (n = 0; n < 4; n++) {
      for (t = 0; t < MAX_SIMULTANEOUS_BULLETS; t++) {
        shotdirection[n][t] = 0;
      }
      ouch[n] = 0;
      status[n] = DEAD;
      energy[n] = 0;
    }

    n = 0;

    cpct_drawStringM1_f("Play with joysticks", (u8*) 0xc564, 2, 0);
    cpct_drawStringM1_f("and MultiPlay device", (u8*) 0xc5b3, 2, 0);
    cpct_drawStringM1_f("or Q A O P + space", (u8*) 0xc605, 2, 0);
    cpct_drawStringM1_f("Press fire to start", (u8*) 0xc654, 2, 0);

    a = 0; // Next player initialization

    cpct_akp_musicInit((u8*) 0x40); // Music

    while (menuscreen) {
      cpct_akp_musicPlay();
      cpct_scanKeyboard();
      if (cpct_isKeyPressed(Key_T)) { // Turbo
        normalspeed = 0;
        cpct_setBorder(palette[1]);
      }
      if (cpct_isKeyPressed(Key_N)) { // Normal speed
        normalspeed = 1;
        cpct_setBorder(palette[0]);
      }
      if (pause) {
        __asm__("halt");
        __asm__("halt");
        __asm__("halt");
        __asm__("halt");
        pause--;
      } else {
        if (count % 8 == 0) {
          cpct_drawCharM1_f((u8*) SCROLLCHARPOS, 3, 0, text[nextChar]);
          if (++nextChar == textlen) nextChar = 0;
          else if (text[nextChar] == '$') {
            pause = 128;
            nextChar++;
          }
        }
        count++;
        AxelayScroller();
      }
      cpct_waitVSYNC();
      checkJoins();
    }

    // Fade music down
    for (n = 0; n < 160; n++) {
      cpct_akp_setFadeVolume(n / 10);
      cpct_akp_musicPlay();
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      cpct_waitVSYNC();
    }
    //cpct_akp_stop();

    cpct_clearScreen(0);

    // Draw screen
    cpct_drawSprite(topleft, (u8*) 0xc000, 10, 56);
    cpct_drawSprite(topcenter, (u8*) 0xc00a, 20, 56);
    cpct_drawSprite(topcenter, (u8*) 0xc01e, 20, 56);
    cpct_drawSprite(topcenter, (u8*) 0xc032, 20, 56);
    cpct_drawSprite(topright, (u8*) 0xc046, 10, 56);
    cpct_drawSprite(bottomleft, (u8*) 0xc640, 8, 40);
    cpct_drawSprite(bottom1, (u8*) 0xc64c, 16, 40);
    cpct_drawSprite(bottom2, (u8*) 0xc660, 16, 40);
    cpct_drawSprite(bottom3, (u8*) 0xc674, 16, 40);
    cpct_drawSprite(bottomright, (u8*) 0xc688, 8, 40);
    for (n = 0; n < 4; n++) {
      cpct_drawSprite(platformempty, row[n][184], TILEWIDTH, TILEHEIGHT);
      drawEnergy(n);
      drawShield(n);
      drawBullets(n);
    }

    // Player 1
    //cpct_drawStringM1_f("Press", (u8*) 0xc3c5, 2, 0);
    //cpct_drawStringM1_f("fire!", (u8*) 0xc415, 2, 0);
    // Player 2
    cpct_drawStringM1_f("Press", (u8*) 0xc3d9, 2, 0);
    cpct_drawStringM1_f("fire!", (u8*) 0xc429, 2, 0);
    // Player 3
    cpct_drawStringM1_f("Press", (u8*) 0xc3ed, 2, 0);
    cpct_drawStringM1_f("fire!", (u8*) 0xc43d, 2, 0);
    // Player 4
    cpct_drawStringM1_f("Press", (u8*) 0xc401, 2, 0);
    cpct_drawStringM1_f("fire!", (u8*) 0xc451, 2, 0);

    weHaveAWinner = 0;
    initialize(a - 1); // Re-initialize the first player to draw him again

    cpct_akp_SFXInit((u8*) 5877); // Sound effects
    cpct_akp_musicInit((u8*) 5877); // Sound effects

    cpct_akp_setFadeVolume(0);

    for (n = 4; n > 0; n--) {
      if (n < 4) cpct_akp_SFXPlay(5, 15, 40, 0, 0, AY_CHANNEL_B);
      screen1(countdown[n]);
      screen2(countdown[n]);
      screen3(countdown[n]);
      for (t = 80; t > 0; t--) {
        checkJoins();
        __asm__("halt");
        __asm__("halt");
        __asm__("halt");
        __asm__("halt");
        cpct_akp_musicPlay();
        cpct_waitVSYNC();
      }
    }
    if (status[PLAYER1] + status[PLAYER2] + status[PLAYER3] + status[PLAYER4] == 1) weHaveAWinner = 1;
    else {
      cpct_akp_SFXPlay(5, 15, 52, 0, 0, AY_CHANNEL_B);
      cpct_akp_musicPlay();
      screen1(countdown[0]);
      screen2(countdown[0]);
      screen3(countdown[0]);
      cpct_waitVSYNC();
      cpct_akp_musicPlay();
    }

    cpct_drawStringM1_f("     ", (u8*) 0xc3c5, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc415, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc3d9, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc429, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc3ed, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc43d, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc401, 2, 0);
    cpct_drawStringM1_f("     ", (u8*) 0xc451, 2, 0);
    cpct_waitVSYNC();

    while (weHaveAWinner == 0 && status[PLAYER1] + status[PLAYER2] + status[PLAYER3] + status[PLAYER4] > 0) {
      count++;
      cpct_scanKeyboard();
      handleLogic(PLAYER1);
      handleLogic(PLAYER2);
      handleLogic(PLAYER3);
      handleLogic(PLAYER4);
      handleShot(PLAYER1);
      handleShot(PLAYER2);
      handleShot(PLAYER3);
      handleShot(PLAYER4);
      cpct_akp_musicPlay();
      cpct_waitVSYNC();
      if (normalspeed) {
        screen1(countdown[0]);
        screen2(countdown[0]);
        screen3(countdown[0]);
        handleShot(PLAYER1);
        handleShot(PLAYER2);
        handleShot(PLAYER3);
        handleShot(PLAYER4);
        cpct_akp_musicPlay();
        cpct_waitVSYNC();
      }
    }

    // Someone won the game! Yay! (or everyone died)

    // Redraw all players (that still has energy),
    // in case one of them were invisible (due to being hit) just prior to game over
    for (n = 0; n < 4; n++) {
      if (energy[n] > 0) {
        cpct_drawSprite(rocket[n], row[n][yposition[n]], TILEWIDTH, TILEHEIGHT);
      }
    }

    if (weHaveAWinner) {

      cpct_drawStringM1_f("Winner: ", (u8*) (0xe000 + 80 * 12) + 30, 3, 0);
      cpct_drawSprite(rocket[weHaveAWinner - 1], (u8*) (0xc000 + 80 * 12) + 46, TILEWIDTH, TILEHEIGHT);
    } else {
      cpct_drawStringM1_f("Draw", (u8*) (0xe000 + 80 * 12) + 36, 3, 0);
    }

    for (n = 100; n > 0; n--) {
      cpct_akp_musicPlay(); // Continue playing sound-effects for a while
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      __asm__("halt");
      cpct_waitVSYNC();
    }

    cpct_akp_stop();
  }
}

